import React, { useRef } from 'react';
import { Link, useLocation } from 'react-router-dom';
import type { Deal, DealStatus } from '../types';
import { COLUMNS } from '../constants';
import { Phone, Mail, Plus, Camera } from 'lucide-react';
import html2canvas from 'html2canvas';

interface CRMBoardProps {
  deals: Deal[];
}

const DealCard: React.FC<{ deal: Deal, borderColor: string }> = ({ deal, borderColor }) => {
  const cardRef = useRef<HTMLDivElement>(null);
  const location = useLocation();

  const handleScreenshot = async () => {
    if (!cardRef.current) return;

    try {
      const canvas = await html2canvas(cardRef.current, {
        backgroundColor: '#ffffff',
        scale: 4, // 4x scale for sharp, high-res output
        useCORS: true,
        logging: false,
        onclone: (clonedDoc) => {
          // Identify the specific card in the clone
          const clonedCard = clonedDoc.querySelector(`[data-card-id="${deal.id}"]`);
          if (clonedCard) {
            // Hide the entire action icon area and activity badge in the screenshot
            const actions = clonedCard.querySelector('.action-icons-area');
            const badge = clonedCard.querySelector('.activity-badge');
            if (actions) (actions as HTMLElement).style.display = 'none';
            if (badge) (badge as HTMLElement).style.display = 'none';
          }
        }
      });

      const image = canvas.toDataURL('image/png', 1.0);
      const link = document.createElement('a');
      link.href = image;
      link.download = `CRM-Deal-${deal.id}.png`;
      link.click();
    } catch (error) {
      console.error('Snapshot failed:', error);
    }
  };

  return (
    <Link
      to={`/crm/deal/${deal.id}`}
      state={{ backgroundLocation: location }}
      className="block no-underline"
    >
      <div
        ref={cardRef}
        data-card-id={deal.id}
        style={{ borderLeft: `5px solid ${borderColor}` }}
        className="bg-white p-4 mb-3 flex flex-col group rounded-[12px] shadow-sm relative transition-all hover:shadow-md border border-slate-100 cursor-pointer"
      >
        {/* Top Section */}
        <div className="flex justify-between items-start mb-2">
          <div className="pr-12">
            <h3 className="text-[14px] font-[500] text-[#333] leading-tight mb-1">
              {deal.title}
            </h3>
          </div>
          <div className="action-icons-area flex flex-col items-center gap-2.5 absolute right-3.5 top-3.5 z-10 opacity-70 group-hover:opacity-100 transition-opacity">
            <span className="activity-badge w-6 h-6 bg-[#f1f5f9] text-[#64748b] text-[11px] font-[800] rounded-full flex items-center justify-center border border-slate-200">
              {deal.activityCount}
            </span>
            <div className="flex flex-col gap-2.5 items-center">
              <button
                onClick={(e) => e.preventDefault()}
                className="text-[#3c82f6] hover:scale-125 transition-transform"><Phone size={14} strokeWidth={2.5} /></button>
              <button
                onClick={(e) => e.preventDefault()}
                className="text-slate-400 hover:text-indigo-600 hover:scale-125 transition-transform"><Mail size={14} strokeWidth={2.5} /></button>
              <button
                onClick={(e) => {
                  e.preventDefault();
                  e.stopPropagation();
                  handleScreenshot();
                }}
                className="text-slate-400 hover:text-emerald-500 hover:scale-125 transition-transform mt-1 pt-1 border-t border-slate-100"
                title="Take Screenshot"
              >
                <Camera size={14} strokeWidth={2.5} />
              </button>
            </div>
          </div>
        </div>

        {/* Amount and Type */}
        <div className="mb-3">
          <div className="text-[15px] text-[#2c3e50]  font-[400]">
            {deal.currency}{deal.amount}
          </div>
          <div className="text-[#3498db] text-[13px] font-[500] ">
            {deal.type}
          </div>
          <div className="text-[#95a5a6] text-[12px]">
            {deal.dateText}
          </div>
        </div>

        {/* Responsible Person */}
        <div className="mb-3">
          <div className="text-[#95a5a6] text-[12px] mb-1.5 font-[400]">Responsible person</div>
          <div className="flex items-center gap-2">
            <div className="w-7 h-7 rounded-full overflow-hidden shrink-0 border border-slate-200">
              <img
                src={deal.responsibleAvatar || `https://api.dicebear.com/7.x/avataaars/svg?seed=${deal.responsiblePerson}&backgroundColor=b6e3f4`}
                alt="avatar"
                className="w-full h-full object-cover"
              />
            </div>
            <span className="text-[#3498db] text-[13px] font-[500] line-clamp-1">
              {deal.responsiblePerson}
            </span>
          </div>
        </div>

        {/* Modified On */}
        <div className="mb-3">
          <div className="text-[#95a5a6] text-[12px]  font-[400]">Modified on</div>
          <div className="text-[#333] text-[13px] font-[400]">
            {deal.modifiedDate}
          </div>
        </div>
        {/* Task */}
        <div className="mb-5">
          <div className="text-[#95a5a6] text-[12px] mb-1.5 font-[400]">Task</div>
          <span className={`px-3 py-1 rounded-[10px] text-[11px] font-[700] uppercase tracking-wide ${deal.taskStatus === 'FINISHED'
            ? 'bg-[#eef5e6] text-[#7a9b5a]'
            : 'bg-blue-50 text-blue-600'
            }`}>
            {deal.taskStatus}
          </span>
        </div>

        {/* Footer */}
        <div className="flex items-center gap-1.5 text-[#95a5a6] cursor-pointer hover:text-slate-600 transition-colors w-fit">
          <Plus size={16} strokeWidth={2} />
          <span className="text-[12px] font-[400]">+ Activity</span>
        </div>
      </div>
    </Link>
  );
};

export const CRMBoard: React.FC<CRMBoardProps> = ({ deals }) => {
  const getDealsByStatus = (status: DealStatus) => deals.filter(d => d.status === status);
  const location = useLocation();

  return (
    <div className="flex flex-1 overflow-x-auto overflow-y-hidden custom-scrollbar px-6 pb-6 gap-0 select-none items-start">
      {COLUMNS.map((col) => {
        const columnDeals = getDealsByStatus(col.id);

        return (
          <div key={col.id} className="min-w-[280px] w-[280px] flex flex-col max-h-[calc(100vh-180px)] min-h-[calc(100vh-180px)] rounded-t-[5px] border-l border-r border-white/10 border-dashed px-2">
            {/* Stage Header - Chevron Style */}
            <div className="column-head-chevron mb-3 shadow-lg" style={{ backgroundColor: col.color }}>
              <span className="truncate pr-4 text-white font-[500] text-[12px] uppercase">{col.title}</span>
              <span className="bg-black/20 px-2 py-0.5 rounded text-[10px] font-bold shrink-0 text-white uppercase">
                {columnDeals.length}
              </span>
            </div>

            {/* Stage Add Action */}
            <div className="px-3 w-full mb-3 flex justify-between items-center text-white drop-shadow-sm">
              <Link
                to="/crm/add"
                state={{ backgroundLocation: location }}
                className="w-full"
              >
                <button className="p-1.5 w-full flex items-center gap-2 text-[12px] font-bold justify-center rounded-[6px] bg-white/15 hover:bg-white/30 transition-all text-white border border-white/20">
                  <Plus size={14} strokeWidth={3} /> Add Deal
                </button>
              </Link>
            </div>


            {/* Scrollable Column */}
            <div className="flex-1 overflow-y-auto custom-scrollbar px-1 pb-10 space-y-2">
              {columnDeals.map(deal => <DealCard key={deal.id} deal={deal} borderColor={col.color} />)}
              {columnDeals.length === 0 && (
                <div className="h-40 border-2 border-dashed border-white/5 rounded-[16px] flex flex-col items-center justify-center text-white/20 font-bold uppercase tracking-widest text-[10px]">
                  Empty Stage
                </div>
              )}
            </div>
          </div>
        );
      })}
    </div>
  );
};