import React, { useEffect, useState } from 'react';
import { useNavigate } from 'react-router-dom';
import { X, Link as LinkIcon, Minimize2, ExternalLink } from 'lucide-react';

interface DrawerProps {
    children: React.ReactNode;
    title?: string;
    onClose?: () => void;
}

const Drawer: React.FC<DrawerProps> = ({ children, title, onClose }) => {
    const navigate = useNavigate();
    const [isOpen, setIsOpen] = useState(false);

    useEffect(() => {
        // Trigger animation after mount
        const timer = setTimeout(() => setIsOpen(true), 10);

        // Escape key listener
        const handleEsc = (event: KeyboardEvent) => {
            if (event.key === 'Escape') handleClose();
        };
        window.addEventListener('keydown', handleEsc);

        return () => {
            clearTimeout(timer);
            window.removeEventListener('keydown', handleEsc);
        };
    }, []);

    const handleClose = () => {
        setIsOpen(false);
        // Wait for animation to finish before calling onClose or navigating back
        setTimeout(() => {
            if (onClose) {
                onClose();
            } else {
                // If we can't go back (direct hit), go to parent /crm
                if (window.history.length > 1) {
                    navigate(-1);
                } else {
                    navigate('/crm');
                }
            }
        }, 400);
    };

    return (
        <div className="fixed inset-0 z-[100] flex justify-end pt-2 ">
            {/* Backdrop */}
            <div
                className={`absolute inset-0 bg-black/60 backdrop-blur-[3px] transition-opacity duration-500 ease-in-out ${isOpen ? 'opacity-100' : 'opacity-0'}`}
                onClick={handleClose}
            />

            <div
                style={{ borderRadius: '10px 0 0 0' }}
                className={`
                    relative w-full max-w-5xl bg-[#f8fafc] h-full
                    shadow-[-20px_0_60px_-15px_rgba(0,0,0,0.3)] 
                    border-l border-y border-white/20 overflow-visible
                    transition-all duration-700 ease-[cubic-bezier(0.16,1,0.3,1)] transform
                    ${isOpen
                        ? 'translate-y-0 opacity-100'
                        : 'translate-y-full opacity-0'
                    }
                `}
            >
                {/* Vertical Action Icons (Left Edge - Vertically Centered) */}
                <div className="absolute -left-[55px] top-[120px] -translate-y-1/2 flex flex-col gap-2 z-20">
                    <button
                        onClick={handleClose}
                        className="w-[54px] h-[38px] bg-[#3ea1db] text-white flex items-center justify-center rounded-l-[30px] shadow-xl border-y border-l border-white/20 hover:bg-[#3491c7] transition-all"
                    >
                        <X size={18} strokeWidth={2.5} />
                    </button>
                    <button className="w-[54px] h-[38px] bg-[#3ea1db] text-white flex items-center justify-center rounded-l-[30px] shadow-xl border-y border-l border-white/20 hover:bg-[#3491c7] transition-all">
                        <LinkIcon size={18} strokeWidth={2.5} />
                    </button>
                    <button className="w-[54px] h-[38px] bg-[#3ea1db] text-white flex items-center justify-center rounded-l-[30px] shadow-xl border-y border-l border-white/20 hover:bg-[#3491c7] transition-all">
                        <Minimize2 size={18} strokeWidth={2.5} />
                    </button>
                    <button className="w-[54px] h-[38px] bg-[#3ea1db] text-white flex items-center justify-center rounded-l-[30px] shadow-xl border-y border-l border-white/20 hover:bg-[#3491c7] transition-all">
                        <ExternalLink size={18} strokeWidth={2.5} />
                    </button>
                </div>
                {/* Header */}
                <div className="flex items-center justify-between px-10 py-8 border-b border-slate-200 bg-white/50 backdrop-blur-2xl sticky top-0 z-10 rounded-tl-[30px]">
                    <div>
                        <h2 className="text-3xl font-black text-slate-800 tracking-tight uppercase leading-none">{title || 'Details'}</h2>
                        <div className="w-16 h-1.5 bg-indigo-500 rounded-full mt-3"></div>
                    </div>
                </div>

                {/* Content */}
                <div className="h-[calc(100%-110px)] overflow-y-auto custom-scrollbar px-10 py-8">
                    {children}
                </div>
            </div>
        </div>
    );
};

export default Drawer;
